--[[---------------------------------------------------------------------------
	Chocolatier Two Simulator: Ports
	Copyright (c) 2006-2007 Big Splash Games, LLC. All Rights Reserved.
--]]---------------------------------------------------------------------------

-- Port class definition
LPort =
{
	__tostring = function(t) return "{Port:" .. t.name .. "}" end,
	_ByName = {},
	_ByIndex = {},
	_Hidden = {},
	Routes = {},
	
	ambient = "sfx/port_loop.ogg",
}

-------------------------------------------------------------------------------
-- Functions for data description (including port layout)

function DefinePort(t)
	t = t or {}
	return LPort:new(t)
end

function SpriteLayer(t)
	local s = t
	if type(s.image) == "string" then s.image = "ports/"..s.image end
	return
	{
		PrepareForLayout = function()
			if type(s.imagelist) == "table" then
				s.image = "ports/" .. s.imagelist[bsgRandom(1,table.getn(s.imagelist))]
			end
			if (not s.random) or (bsgRandom(1,100) <= s.random) then
				return function() bsgAddSprite(s) end
			else
				return function() end
			end
		end,
		startweek = s.startweek,
		endweek = s.endweek,
	}
	
--	return function()
--		return function() bsgAddSprite(s) end
--	end
end

function PlayerLogoRollover()
	StopPulsing()
	return MakeRollover
	{
		x=0,y=0, color=PopupColor, inset=2,
		AppendStyle { font = popupFont, flags=kHAlignLeft+kVAlignTop },
		TightText { x=0,y=0, w=800,h=400, label="logo_edit" },
	}
end

local function DoPlayerLogo()
	DoModal("ui/logoedit_basic.lua")
	SwapToModal("ui/portview.lua")
	StopPulsing()
end

function PlayerLogo(t)
	local s = t
	s.image = ".LOGO"
	if s.target then s.target = LBuilding:ByName(s.target) end
	return function()
		s.logo = gSim.logo
		s.contents = "PlayerLogoRollover()"
		s.command = DoPlayerLogo
		return function()
			if s.logo and gSim.rank > 0 and s.target and s.target.owned then
				s.labelx = s.x - 120
				s.labely = s.y - 10
				bsgAddSprite(s)
			end
		end
	end
end

function BaumeisterSign(t)
	local s = t
	return function()
		local n = LQuest:Variable("bcsign")
		if n < 6 then
			-- This does not get reset between players, make sure it's accurate...
			s.image = "ports/baumeister"..tostring(n)
			s.logo = nil
			s.contents = nil
			s.command = nil
			s.scale = 1
			s.y = 55+100/2
			return function()
				bsgAddSprite(s)
			end
		else
			s.image = ".LOGO"
			s.logo = gSim.logo
			s.contents = "PlayerLogoRollover()"
			s.command = DoPlayerLogo
			s.scale = 175 / 350
--			s.y = s.y - 10			-- Since this table is permanent, this makes a permanent (cumulative) change...
			s.y = 55+100/2 - 10
			return function()
				bsgAddSprite(s)
			end
		end
	end
end

function Fireworks()
	return function()
		return function()
			local n = LQuest:Variable("bcsign")
			if n > 5 and StartFireworks then
				StartFireworks()
			end
		end
	end
end

-------------------------------------------------------------------------------
-- "static" functions to access global Port lists

function LPort:AllPorts()
	return bsgArrayIterator(self._ByIndex)
end

function LPort:ByName(portName)
	return self._ByName[portName]
end

-------------------------------------------------------------------------------

function LPort:GetRoute(a, b)
	-- Somewhat confusing but useful
	local fromName = a
	local toName = b
	if not b then
		fromName = self.name
		toName = a
	end
	
	local route = self.Routes[fromName]
	return route[toName]
end

function LPort:GetRoutes()
	if self.name then return self.Routes[self.name]
	else return self.Routes
	end
end

-------------------------------------------------------------------------------
-- "constructor"

function LPort:new(t)
	-- Standard object creation...
	t = t or {} setmetatable(t, self) self.__index = self

	-- Check parameters
	if not t.name then
		bsgDevWarning("Port defined with no name")
	elseif self._ByName[t.name] then
		bsgDevWarning("Port " .. t.name .. " already defined")
	else
		-- If hidden, port is not available
		if t.hidden then
			t.available = false
			self._Hidden[t.name] = true
		end
		
		-- If not specifically un-available, port is available
		if t.available == nil then t.available = true end
		
		-- Provide port layout, even if empty
		t.layout = t.layout or {}
		
		-- Prepare routes
		self.Routes[t.name] = {}
		
		-- Keep global tables
		self._ByName[t.name] = t
		table.insert(self._ByIndex, t)

		return t
	end
	
	return nil
end

-------------------------------------------------------------------------------
-- Audio

function LPort:SetAmbient()
	SetAmbient(self.ambient)
	
	-- In Hidden ports, cut music and just go for ambients...
	if self.hidden then SetMusicState("silent")
	else SetMusicState("normal")
	end
end

-------------------------------------------------------------------------------
-- Data post-processing

function LPort:PostProcessAll()
	for _,port in ipairs(self._ByIndex) do
		-- Move port locations to the port itself, portlocations.lua just handy for development
		local loc = self.Locations[port.name]
		port.mapx = loc[1] or 10
		port.mapy = loc[2] or 10
		port.labelpos = loc[3] or kHAlignRight

		-- Build completely cross-referenced route table
		for _,dest in ipairs(self._ByIndex) do
			if port ~= dest then BuildCompleteRoute(port.name, dest.name) end
		end
		
		-- Cross-reference buildings to ports
		for _,b in ipairs(port.layout) do
			if b.SetPort then b:SetPort(port) end
		end
	end
end

-------------------------------------------------------------------------------
-- Popups

function LPort:RolloverTarget()
	return "(LPort:ByName('"..self.name.."')):MapRollover()"
end

function LPort:MapRollover()
	if gTravelActive then return nil end
	
	-- Build rollover label text
	local label = { "#"..GetString(self.name) }

	-- Travel information
	if not self.available then
		if gSim.mode == "free" then
			table.insert(label, GetString("travel_freedisabled"))
		else
			table.insert(label, GetString("travel_disabled"))
		end
	elseif gSim.port ~= self then
		local route = gSim.port:GetRoute(self.name)
		
		local time = route.totalWeeks
		if LQuest:Variable("airship") > 0 then time = bsgFloor(time/2 + .5) end
		
		table.insert(label, GetString("travel_time", tostring(time)))
		
		-- TODO: TRAVEL PRICING!
		local price = route.price or (route.totalWeeks * gSim.travelCost)
		if LQuest:Variable("airship") > 0 then price = 0 end
		if price > 0 then
			table.insert(label, GetString("travel_price", bsgDollars(price)))
			if gSim.money < price then
				table.insert(label, GetString("travel_expensive"))
			end
		end
		if price <= gSim.money then
			table.insert(label, GetString("travel_click"))
		end
	else
		table.insert(label, GetString("travel_here"))
	end
	
	label = table.concat(label,"\n")

	-- Flip tag if it will hit the right edge of the screen
	local hflip = false
	if self.mapx + ui.travelTagWidth >= ui.screenWidth then
		hflip = true
		label = Text { x=5,y=5,w=kMax-51,h=kMax-5, label=label, font=travelTagFont, flags=kVAlignCenter+kHAlignLeft, }
	else
		label = Text { x=51,y=5,w=kMax-5,h=kMax-5, label=label, font=travelTagFont, flags=kVAlignCenter+kHAlignLeft, }
	end

	return MakeDialog
	{
		Bitmap
		{
			fit=true,
			x=0,y=0, image="image/traveltag", hflip=hflip,
			label
		}
	}
end

-------------------------------------------------------------------------------
-- Reset

function LPort:ResetAll()
	for p in LPort:AllPorts() do
		p.visited = nil
		p.available = nil
		p.hidden = self._Hidden[p.name]
		p.factoryOwned = nil
		p.shopOwned = nil
		p.labOwned = nil
	end
end

-------------------------------------------------------------------------------
-- Load and Save

function LPort:BuildSaveTable()
	local t = {}
	for _,p in ipairs(self._ByIndex) do
		if p.available then
			t[p.name] = p.visited or false
		end
	end
	return t
end

function LPort:LoadSaveTable(t)
	-- If it's in the list at all, it's available
	for name,data in pairs(t) do
		local p = LPort:ByName(name)
		if p then
			p.available = true
			p.visited = data
		end
	end
end
